"use client";
import { useState } from "react";
import Link from "next/link";
import { ArrowLeft, AlertCircle, MailSearch } from "lucide-react";
import { cn } from "@/lib/utils";
import { validateFindEmail } from "@/lib/auth-validation";

export default function ForgotEmail() {
  const [step, setStep] = useState(1); // 1: Recovery Info, 2: Name, 3: OTP, 4: Results
  const [error, setError] = useState("");
  const [formData, setFormData] = useState({
    recoveryInput: "",
    firstName: "",
    lastName: "",
    otp: ""
  });

  // Mock result for demo
  const foundEmails = ["jo****n@gmail.com", "j.d**e@indzs.com"];

  const handleNext = (e: React.FormEvent) => {
    e.preventDefault();
    const validationError = validateFindEmail(step, formData);
    if (validationError) {
      setError(validationError);
      return;
    }
    setError("");
    setStep(step + 1);
  };

  return (
    <div className="min-h-screen bg-background flex flex-col items-center justify-center p-4 w-full">
      <div className="w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] border border-border shadow-2xl overflow-hidden flex flex-col md:flex-row min-h-[500px]">
        
        {/* Left Side */}
        <div className="flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
          <div className="space-y-6">
            <div className="text-2xl font-bold tracking-tighter">
                <span className="text-[#4285F4]">I</span><span className="text-[#EA4335]">n</span><span className="text-[#FBBC05]">d</span><span className="text-[#34A853]">z</span><span className="text-[#4285F4]">s</span>
            </div>
            <div className="space-y-2">
              <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
                {step === 1 && "Find your email"}
                {step === 2 && "What's your name?"}
                {step === 3 && "Get a code"}
                {step === 4 && "Account found"}
              </h1>
              <p className="text-muted-foreground text-lg font-light max-w-sm">
                {step === 1 && "Enter your recovery phone number or email address."}
                {step === 2 && "Enter the name on your Indzs Account."}
                {step === 3 && `We'll send a verification code to ${formData.recoveryInput}.`}
                {step === 4 && "Select an account to sign in."}
              </p>
            </div>
          </div>
          {step > 1 && step < 4 && (
            <button onClick={() => setStep(step - 1)} className="flex items-center gap-2 text-sidebar-primary hover:bg-sidebar-primary/10 w-fit px-4 py-2 rounded-full transition-all">
              <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
            </button>
          )}
        </div>

        {/* Right Side */}
        <div className="flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">
          <form onSubmit={handleNext} className="space-y-6 animate-in fade-in slide-in-from-right-4 duration-500">
            
            {/* Step 1: Recovery Input */}
            {step === 1 && (
              <div className="relative group">
                <input
                  type="text" required
                  className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 text-foreground outline-none transition-all peer placeholder-transparent", error && "border-destructive")}
                  placeholder="Phone or recovery email"
                  value={formData.recoveryInput}
                  onChange={(e) => setFormData({...formData, recoveryInput: e.target.value})}
                />
                <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:text-sidebar-primary peer-focus:bg-card px-2 peer-[:not(:placeholder-shown)]:-top-3">
                  Phone or recovery email
                </label>
              </div>
            )}

            {/* Step 2: Name Verification */}
            {step === 2 && (
              <div className="space-y-4">
                <div className="relative group">
                  <input
                    type="text" required
                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                    placeholder="First"
                    onChange={(e) => setFormData({...formData, firstName: e.target.value})}
                  />
                  <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card px-1 peer-[:not(:placeholder-shown)]:-top-3">First name</label>
                </div>
                <div className="relative group">
                  <input
                    type="text" required
                    className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent"
                    placeholder="Last"
                    onChange={(e) => setFormData({...formData, lastName: e.target.value})}
                  />
                  <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card px-1 peer-[:not(:placeholder-shown)]:-top-3">Last name</label>
                </div>
              </div>
            )}

            {/* Step 3: OTP */}
            {step === 3 && (
               <div className="relative">
               <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">G -</span>
               <input
                   type="text" maxLength={6} autoFocus
                   className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-12 pr-5 py-4 text-foreground tracking-[0.5em] font-bold outline-none", error && "border-destructive")}
                   value={formData.otp}
                   onChange={(e) => setFormData({ ...formData, otp: e.target.value.replace(/\D/g, "") })}
               />
           </div>
            )}

            {/* Step 4: Results */}
            {step === 4 && (
              <div className="space-y-3">
                {foundEmails.map((email, idx) => (
                  <Link 
                    key={idx} 
                    href={`/login?email=${email}`}
                    className="flex items-center p-4 border border-border rounded-2xl hover:bg-sidebar-primary/5 transition-colors group"
                  >
                    <div className="w-10 h-10 rounded-full bg-sidebar-primary/10 flex items-center justify-center text-sidebar-primary mr-4">
                        {formData.firstName[0]}
                    </div>
                    <div>
                        <p className="font-medium text-foreground">{formData.firstName} {formData.lastName}</p>
                        <p className="text-sm text-muted-foreground">{email}</p>
                    </div>
                  </Link>
                ))}
              </div>
            )}

            {error && (
                <div className="flex items-center gap-2 text-destructive text-sm px-2 animate-in fade-in slide-in-from-top-1">
                    <AlertCircle size={16} />
                    <span>{error}</span>
                </div>
            )}

            <div className="flex items-center justify-end pt-8">
              {step < 4 ? (
                <button type="submit" className="bg-sidebar-primary text-white font-semibold px-12 py-3 rounded-2xl transition-all">
                   Next
                </button>
              ) : (
                <Link href="/login" className="bg-sidebar-primary text-white font-semibold px-12 py-3 rounded-2xl transition-all">
                  Done
                </Link>
              )}
            </div>
          </form>
        </div>
      </div>
    </div>
  );
}